import Foundation

/// Numerical integration engine for calculating life expectancy from mortality functions
/// Implements Simpson's rule and Gaussian quadrature for actuarial calculations
class IntegrationEngine {
    
    /// Integration methods available
    enum IntegrationMethod {
        case simpsonsRule
        case gaussianQuadrature
        case trapezoidalRule
        case adaptiveQuadrature
    }
    
    /// Integration parameters for accuracy control
    struct IntegrationParams {
        let method: IntegrationMethod
        let stepSize: Double
        let maxAge: Double
        let tolerance: Double
        let maxIterations: Int
        
        static let `default` = IntegrationParams(
            method: .simpsonsRule,
            stepSize: 0.1,
            maxAge: 150.0,
            tolerance: 1e-6,
            maxIterations: 10000
        )
        
        static let highPrecision = IntegrationParams(
            method: .adaptiveQuadrature,
            stepSize: 0.01,
            maxAge: 150.0,
            tolerance: 1e-8,
            maxIterations: 50000
        )
    }
    
    /// Calculate life expectancy by integrating survival function
    /// e(x) = ∫₀^∞ S(x+t) dt where S(t) is survival probability
    func calculateLifeExpectancy(
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel,
        params: IntegrationParams = .default
    ) -> Double {
        
        switch params.method {
        case .simpsonsRule:
            return integrateSurvivalFunctionSimpson(
                currentAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel,
                params: params
            )
        case .gaussianQuadrature:
            return integrateSurvivalFunctionGaussian(
                currentAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel,
                params: params
            )
        case .trapezoidalRule:
            return integrateSurvivalFunctionTrapezoidal(
                currentAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel,
                params: params
            )
        case .adaptiveQuadrature:
            return integrateSurvivalFunctionAdaptive(
                currentAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel,
                params: params
            )
        }
    }
    
    /// Simpson's rule integration of survival function
    private func integrateSurvivalFunctionSimpson(
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel,
        params: IntegrationParams
    ) -> Double {
        
        let stepSize = params.stepSize
        let maxAge = params.maxAge
        let steps = Int((maxAge - currentAge) / stepSize)
        
        guard steps > 0 else { return 0.0 }
        
        var lifeExpectancy: Double = 0.0
        
        // Simpson's rule: ∫f(x)dx ≈ (h/3)[f(x₀) + 4f(x₁) + 2f(x₂) + 4f(x₃) + ... + f(xₙ)]
        for i in 0...steps {
            let age = currentAge + Double(i) * stepSize
            let survivalProbability = calculateSurvivalProbability(
                toAge: age,
                fromAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel
            )
            
            let coefficient: Double
            if i == 0 || i == steps {
                coefficient = 1.0
            } else if i % 2 == 1 {
                coefficient = 4.0
            } else {
                coefficient = 2.0
            }
            
            lifeExpectancy += coefficient * survivalProbability * stepSize
            
            // Early termination for negligible survival probabilities
            if survivalProbability < params.tolerance {
                break
            }
        }
        
        return lifeExpectancy / 3.0
    }
    
    /// Gaussian quadrature integration (more accurate for smooth functions)
    private func integrateSurvivalFunctionGaussian(
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel,
        params: IntegrationParams
    ) -> Double {
        
        // 5-point Gaussian quadrature nodes and weights
        let nodes: [Double] = [-0.9061798459, -0.5384693101, 0.0, 0.5384693101, 0.9061798459]
        let weights: [Double] = [0.2369268851, 0.4786286705, 0.5688888889, 0.4786286705, 0.2369268851]
        
        let segments = Int((params.maxAge - currentAge) / params.stepSize)
        let segmentSize = (params.maxAge - currentAge) / Double(segments)
        
        var lifeExpectancy: Double = 0.0
        
        for segment in 0..<segments {
            let a = currentAge + Double(segment) * segmentSize
            let b = currentAge + Double(segment + 1) * segmentSize
            let midpoint = (a + b) / 2.0
            let halfWidth = (b - a) / 2.0
            
            var segmentIntegral: Double = 0.0
            
            for i in 0..<nodes.count {
                let x = midpoint + halfWidth * nodes[i]
                let survivalProb = calculateSurvivalProbability(
                    toAge: x,
                    fromAge: currentAge,
                    healthMetrics: healthMetrics,
                    mortalityModel: mortalityModel
                )
                
                segmentIntegral += weights[i] * survivalProb
            }
            
            lifeExpectancy += segmentIntegral * halfWidth
            
            // Check for convergence
            let midSurvival = calculateSurvivalProbability(
                toAge: midpoint,
                fromAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel
            )
            
            if midSurvival < params.tolerance {
                break
            }
        }
        
        return lifeExpectancy
    }
    
    /// Trapezoidal rule integration (simpler, less accurate)
    private func integrateSurvivalFunctionTrapezoidal(
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel,
        params: IntegrationParams
    ) -> Double {
        
        let stepSize = params.stepSize
        let maxAge = params.maxAge
        var lifeExpectancy: Double = 0.0
        
        var age = currentAge
        var previousSurvival = 1.0 // Survival probability at current age is 1.0
        
        while age < maxAge {
            let nextAge = min(age + stepSize, maxAge)
            let currentSurvival = calculateSurvivalProbability(
                toAge: nextAge,
                fromAge: currentAge,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel
            )
            
            // Trapezoidal rule: area = (h/2)(f(x₀) + f(x₁))
            lifeExpectancy += (stepSize / 2.0) * (previousSurvival + currentSurvival)
            
            previousSurvival = currentSurvival
            age = nextAge
            
            // Early termination
            if currentSurvival < params.tolerance {
                break
            }
        }
        
        return lifeExpectancy
    }
    
    /// Adaptive quadrature with error estimation
    private func integrateSurvivalFunctionAdaptive(
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel,
        params: IntegrationParams
    ) -> Double {
        
        return adaptiveSimpsons(
            from: currentAge,
            to: params.maxAge,
            tolerance: params.tolerance,
            maxDepth: 10,
            currentAge: currentAge,
            healthMetrics: healthMetrics,
            mortalityModel: mortalityModel
        )
    }
    
    /// Recursive adaptive Simpson's rule
    private func adaptiveSimpsons(
        from a: Double,
        to b: Double,
        tolerance: Double,
        maxDepth: Int,
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel
    ) -> Double {
        
        guard maxDepth > 0 else {
            // Fallback to simple Simpson's rule
            return simpleSimpsons(from: a, to: b, currentAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        }
        
        let c = (a + b) / 2.0
        
        let wholeInterval = simpleSimpsons(from: a, to: b, currentAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        let leftHalf = simpleSimpsons(from: a, to: c, currentAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        let rightHalf = simpleSimpsons(from: c, to: b, currentAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        
        let error = abs(wholeInterval - (leftHalf + rightHalf)) / 15.0
        
        if error < tolerance {
            return leftHalf + rightHalf + error // Richardson extrapolation
        } else {
            let leftResult = adaptiveSimpsons(
                from: a, to: c, tolerance: tolerance / 2.0, maxDepth: maxDepth - 1,
                currentAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel
            )
            let rightResult = adaptiveSimpsons(
                from: c, to: b, tolerance: tolerance / 2.0, maxDepth: maxDepth - 1,
                currentAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel
            )
            return leftResult + rightResult
        }
    }
    
    /// Simple Simpson's rule for a single interval
    private func simpleSimpsons(
        from a: Double,
        to b: Double,
        currentAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel
    ) -> Double {
        
        let c = (a + b) / 2.0
        let h = (b - a) / 6.0
        
        let fa = calculateSurvivalProbability(toAge: a, fromAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        let fc = calculateSurvivalProbability(toAge: c, fromAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        let fb = calculateSurvivalProbability(toAge: b, fromAge: currentAge, healthMetrics: healthMetrics, mortalityModel: mortalityModel)
        
        return h * (fa + 4.0 * fc + fb)
    }
    
    /// Calculate survival probability from current age to target age
    private func calculateSurvivalProbability(
        toAge: Double,
        fromAge: Double,
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel
    ) -> Double {
        
        guard toAge >= fromAge else { return 1.0 }
        
        let stepSize = 0.1
        let steps = Int((toAge - fromAge) / stepSize)
        var cumulativeMortality: Double = 0.0
        
        for i in 0..<steps {
            let age = fromAge + Double(i) * stepSize
            let mortality = try! mortalityModel.mortalityRate(at: age, healthMetrics: healthMetrics)
            cumulativeMortality += mortality * stepSize
        }
        
        // Handle remaining fraction
        if steps > 0 {
            let remainingAge = toAge - fromAge - Double(steps) * stepSize
            if remainingAge > 0 {
                let mortality = try! mortalityModel.mortalityRate(at: fromAge + Double(steps) * stepSize, healthMetrics: healthMetrics)
                cumulativeMortality += mortality * remainingAge
            }
        }
        
        // Survival probability = exp(-∫mortality dt)
        return exp(-cumulativeMortality)
    }
    
    /// Comprehensive life expectancy calculation combining all models
    func calculateFinalLifeExpectancy(
        healthMetrics: HealthMetrics,
        baselineLifeExpectancy: Double,
        gompertzMakeham: GompertzMakehamModel,
        coxModel: CoxProportionalHazards,
        riskFactors: [RiskFactor],
        params: IntegrationParams = .default
    ) -> (lifeExpectancy: Double, errorBounds: (lower: Double, upper: Double)) {
        
        // Create composite mortality model
        let compositeMortality = CompositeMortalityModel(
            baseline: baselineLifeExpectancy,
            gompertzMakeham: gompertzMakeham,
            coxModel: coxModel,
            riskFactors: riskFactors
        )
        
        // Calculate life expectancy using specified integration method
        let lifeExpectancy = calculateLifeExpectancy(
            currentAge: 30.0, // Default age for calculation
            healthMetrics: healthMetrics,
            mortalityModel: compositeMortality,
            params: params
        )
        
        // Estimate error bounds using different methods
        let simpsonResult = calculateLifeExpectancy(
            currentAge: 30.0,
            healthMetrics: healthMetrics,
            mortalityModel: compositeMortality,
            params: IntegrationParams(method: .simpsonsRule, stepSize: params.stepSize, maxAge: params.maxAge, tolerance: params.tolerance, maxIterations: params.maxIterations)
        )
        
        let trapezoidalResult = calculateLifeExpectancy(
            currentAge: 30.0,
            healthMetrics: healthMetrics,
            mortalityModel: compositeMortality,
            params: IntegrationParams(method: .trapezoidalRule, stepSize: params.stepSize, maxAge: params.maxAge, tolerance: params.tolerance, maxIterations: params.maxIterations)
        )
        
        // Error bounds based on method differences
        let methodError = abs(simpsonResult - trapezoidalResult) / 2.0
        let lowerBound = max(0, lifeExpectancy - methodError)
        let upperBound = lifeExpectancy + methodError
        
        return (lifeExpectancy, (lowerBound, upperBound))
    }
    
    /// Composite mortality model combining all factors
    private class CompositeMortalityModel: MortalityModel {
        private let baseline: Double
        private let gompertzMakeham: GompertzMakehamModel
        private let coxModel: CoxProportionalHazards
        private let riskFactors: [RiskFactor]
        
        var modelName: String { return "CompositeMortalityModel" }
        var modelVersion: String { return "1.0.0" }
        
        init(baseline: Double, gompertzMakeham: GompertzMakehamModel, coxModel: CoxProportionalHazards, riskFactors: [RiskFactor]) {
            self.baseline = baseline
            self.gompertzMakeham = gompertzMakeham
            self.coxModel = coxModel
            self.riskFactors = riskFactors
        }
        
        func calculateMortality(age: Double, healthMetrics: HealthMetrics) -> Double {
            // Base mortality from Gompertz-Makeham
            let baseMortality = gompertzMakeham.calculateMortality(age: age, gender: .male)
            
            // Risk factor adjustments via Cox model
            let hazardRatio = coxModel.calculateHazardRatio(healthMetrics: healthMetrics, age: age, gender: .male)
            
            // Additional risk factor impacts
            let riskAdjustment = riskFactors.reduce(1.0) { result, riskFactor in
                return result * riskFactor.calculateImpactMultiplier(healthMetrics: healthMetrics)
            }
            
            return baseMortality * hazardRatio * riskAdjustment
        }
        
        func calculateLifeExpectancy(healthMetrics: HealthMetrics) -> Double {
            // This will be calculated through integration
            return baseline
        }
        
        // MARK: - MortalityModel Protocol Methods
        
        func mortalityRate(at age: Double, healthMetrics: HealthMetrics?) throws -> Double {
            guard let healthMetrics = healthMetrics else {
                return try gompertzMakeham.mortalityRate(at: age, healthMetrics: nil)
            }
            return calculateMortality(age: age, healthMetrics: healthMetrics)
        }
        
        func survivalProbability(from currentAge: Double, to targetAge: Double, healthMetrics: HealthMetrics?) throws -> Double {
            return try gompertzMakeham.survivalProbability(from: currentAge, to: targetAge, healthMetrics: healthMetrics)
        }
        
        func lifeExpectancy(at age: Double, healthMetrics: HealthMetrics?) throws -> Double {
            return try gompertzMakeham.lifeExpectancy(at: age, healthMetrics: healthMetrics)
        }
    }
}

/// Extensions for validation and testing
extension IntegrationEngine {
    
    /// Test integration accuracy against known analytical solutions
    func validateIntegrationAccuracy() -> Bool {
        // Test with simple exponential function: ∫₀¹ e^x dx = e - 1 ≈ 1.7183
        let testResult = testExponentialIntegration()
        let expectedResult = exp(1.0) - 1.0
        let relativeError = abs(testResult - expectedResult) / expectedResult
        
        return relativeError < 0.001 // 0.1% tolerance
    }
    
    private func testExponentialIntegration() -> Double {
        // Simplified test using trapezoidal rule
        let steps = 1000
        let stepSize = 1.0 / Double(steps)
        var integral: Double = 0.0
        
        for i in 0...steps {
            let x = Double(i) * stepSize
            let value = exp(x)
            
            let coefficient = (i == 0 || i == steps) ? 1.0 : 2.0
            integral += coefficient * value * stepSize
        }
        
        return integral / 2.0
    }
    
    /// Compare different integration methods for consistency
    func compareIntegrationMethods(
        healthMetrics: HealthMetrics,
        mortalityModel: MortalityModel
    ) -> [String: Double] {
        
        let methods: [(String, IntegrationMethod)] = [
            ("Simpson's Rule", .simpsonsRule),
            ("Trapezoidal Rule", .trapezoidalRule),
            ("Gaussian Quadrature", .gaussianQuadrature)
        ]
        
        var results: [String: Double] = [:]
        
        for (name, method) in methods {
            let params = IntegrationParams(
                method: method,
                stepSize: 0.1,
                maxAge: 120.0,
                tolerance: 1e-6,
                maxIterations: 10000
            )
            
            let result = calculateLifeExpectancy(
                currentAge: 30.0,
                healthMetrics: healthMetrics,
                mortalityModel: mortalityModel,
                params: params
            )
            
            results[name] = result
        }
        
        return results
    }
}